/*
 * JBoss, Home of Professional Open Source
 * Copyright 2005, JBoss Inc., and individual contributors as indicated
 * by the @authors tag. See the copyright.txt in the distribution for a
 * full listing of individual contributors.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of
 * the License, or (at your option) any later version.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, write to the Free
 * Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
 * 02110-1301 USA, or see the FSF site: http://www.fsf.org.
 */
package org.jbpm.jpdl.internal.activity;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.jbpm.api.Execution;
import org.jbpm.api.JbpmException;
import org.jbpm.api.activity.ActivityExecution;
import org.jbpm.pvm.internal.el.Expression;
import org.jbpm.pvm.internal.env.Context;
import org.jbpm.pvm.internal.env.EnvironmentImpl;
import org.jbpm.pvm.internal.env.ExecutionContext;
import org.jbpm.pvm.internal.model.ActivityImpl;
import org.jbpm.pvm.internal.model.Condition;
import org.jbpm.pvm.internal.model.ExecutionImpl;
import org.jbpm.pvm.internal.model.TransitionImpl;

/**
 * @author Maciej Swiderski
 * @author Alejandro Guizar
 */
public class ForEachActivity extends JpdlActivity {

  private String variable;
  private Expression collection;

  private static final long serialVersionUID = 1L;

  public void execute(ActivityExecution execution) throws Exception {
    execute((ExecutionImpl) execution);
  }

  public void execute(ExecutionImpl execution) {
    // resolve collection values
    Collection<?> collection = evaluateCollection(execution);
    ActivityImpl activity = execution.getActivity();

    // get concurrent root
    ExecutionImpl concurrentRoot;
    if (Execution.STATE_ACTIVE_ROOT.equals(execution.getState())) {
      concurrentRoot = execution;
      execution.setState(Execution.STATE_INACTIVE_CONCURRENT_ROOT);
      execution.setActivity(null);
    }
    else if (Execution.STATE_ACTIVE_CONCURRENT.equals(execution.getState())) {
      concurrentRoot = execution.getParent();
      execution.end();
    }
    else {
      throw new AssertionError(execution.getState());
    }

    // evaluate transition condition and create concurrent executions
    TransitionImpl transition = activity.getDefaultOutgoingTransition();
    List<ExecutionImpl> concurrentExecutions = new ArrayList<ExecutionImpl>();
    int index = 1;
    
    //execution context needs to be temporarily replaced to give access to child execution variables
    ExecutionContext originalContext = null;
    EnvironmentImpl environment = EnvironmentImpl.getCurrent();
    if (environment != null) {
      originalContext = (ExecutionContext) environment.removeContext(Context.CONTEXTNAME_EXECUTION);
    }

    try {
      for (Object value : collection) {
        ExecutionImpl concurrentExecution = concurrentRoot.createExecution(Integer
          .toString(index++));
        concurrentExecution.setActivity(activity);
        concurrentExecution.setState(Execution.STATE_ACTIVE_CONCURRENT);
        concurrentExecution.createVariable(variable, value);
  
        // replace the current execution context for expression evaluation purposes
        if (environment != null) {
          environment.setContext(new ExecutionContext(concurrentExecution));
        }
  
        Condition condition = transition.getCondition();
        if (condition == null || condition.evaluate(concurrentExecution)) {
          concurrentExecutions.add(concurrentExecution);
        }
        else {
          concurrentExecution.end();
        }
      }
    }
    finally {
      // reset original execution context after creating concurrent executions
      if (environment != null) {
        environment.setContext(originalContext);
      }
    }

    if (concurrentExecutions.isEmpty()) {
      // if no outgoing transitions should be forked, end this execution
      execution.end();
    }
    else {
      for (ExecutionImpl concurrentExecution : concurrentExecutions) {
        concurrentExecution.take(transition);
        if (concurrentRoot.isEnded()) break;
      }
    }
  }

  private Collection<?> evaluateCollection(ExecutionImpl execution) {
    Object value = collection.evaluate(execution);
    if (value instanceof Collection<?>) {
      // return collection verbatim
      return (Collection<?>) value;
    }
    else if (value instanceof Object[]) {
      // wrap array in list
      return Arrays.asList((Object[]) value);
    }
    else if (value instanceof String) {
      // split string around commas or spaces
      String csv = (String) value;
      return Arrays.asList(csv.split("[,\\s]+"));
    }
    throw new JbpmException("not a collection: " + value);
  }

  public void setVariable(String variable) {
    this.variable = variable;
  }

  public void setCollection(Expression collection) {
    this.collection = collection;
  }

}
